#!/usr/bin/env python3
# Copyright (c) Microsoft. All rights reserved.

# This file contains the the classes for each individual setting

import os
import socket
import subprocess
from abc import ABCMeta, abstractmethod
import mssqlconfhelper
import re
import extensibilityruntime
from uuid import UUID
import urllib
import re
import json
import contextlib

_ = mssqlconfhelper._

class SettingValueType():
    """Setting value type mappings
    """

    boolean = 1
    integer = 2
    string = 3
    dirPath = 4
    traceFlag = 5
    filePath = 6

class SectionForSetting():
    """Section mappings for settings
    """

    traceflag = "traceflag"
    errorlog = "errorlog"
    file_location = "filelocation"
    network = "network"
    hadr = "hadr"
    coredump = "coredump"
    governance = "governance"
    language = "language"
    sqlagent = "sqlagent"
    sqlconnector = "sqlconnector"
    telemetry = "telemetry"
    memory = "memory"
    eula = mssqlconfhelper.eulaConfigSection
    control = "control"
    uncmapping = "uncmapping"
    distributedtransaction = "distributedtransaction"
    extensibility = "extensibility"
    wmi = "wmi"
    licensing = "licensing"

DOMAIN_FORMAT = re.compile(
    r"(?:^(\w{1,255}):(.{1,255})@|^)" # http basic authentication [optional]
    r"(?:(?:(?=\S{0,253}(?:$|:))" # check full domain length to be less than or equal to 253 (starting after http basic auth, stopping before port)
    r"((?:[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?\.)+" # check for at least one subdomain (maximum length per subdomain: 63 characters), dashes in between allowed
    r"(?:[a-z0-9]{1,63})))" # check for top level domain, no dashes allowed
    r"|localhost)" # accept also "localhost" only
    r"(:\d{1,5})?", # port [optional]
    re.IGNORECASE
)
SCHEME_FORMAT = re.compile(
    r"^(http)s?$", # scheme: http(s)
    re.IGNORECASE
)

def validateDirectoryExists(setting, directoryPath):
    """Validate a setting directory path

    Args:
        setting (str): Setting
        directoryPath (str): Directory path

    Returns:
        True if valid, False otherwise
    """

    if not os.path.isdir(directoryPath):
        mssqlconfhelper.printValidationErrorMessage(setting, _("'%s' is not a valid directory") % directoryPath)
        return False
    return True

def validateFileExists(setting, filePath):
    """Validate a file exists

    Args:
        setting (str): Setting
        filePath (str): File path

    Returns:
        True if valid, False otherwise
    """

    if not os.path.isfile(filePath):
        mssqlconfhelper.printValidationErrorMessage(setting, _("'%s' is not a valid file") % filePath)
        return False
    return True

class MssqlRegistrySetting(metaclass=ABCMeta):
    """Abstract Class for implementing a SqlSetting
    """

    def __init__(self, name, environment_variable, setting_value_type, description, section, restart_required, hidden=False, sectionOnly=False):
        self.name = name
        self.environment_variable = environment_variable
        self.setting_value_type = setting_value_type
        self.description = description
        self.section = section
        self.restart_required = restart_required
        self.hidden = hidden
        self.sectionOnly = sectionOnly

    @abstractmethod
    def validateSetting(self, setting_value, validate_verb_used=False, setting_name=""):
        pass

class BooleanSetting(MssqlRegistrySetting):
    """Standard boolean setting
    """

    def __init__(self, name, environment_variable, setting_value_type, description, section, restart_required, true_string="1", false_string="0"):
        super(BooleanSetting, self).__init__(name, environment_variable, setting_value_type, description, section, restart_required)
        self.true_string = true_string.lower()
        self.false_string = false_string.lower()
        self.true_expected_value = self.true_string
        self.false_expected_value = self.false_string
        self.true_string = ("1" if (self.true_string == "true" or self.true_string == "yes") else self.true_string)
        self.false_string = ("0" if (self.false_string == "false" or self.false_string == "no") else self.false_string)

    def validateSetting(self, setting_value, validate_verb_used=False):
        setting_value_lower = setting_value.lower()
        if setting_value_lower != self.true_string and setting_value_lower != self.false_string and setting_value_lower != "true" and setting_value_lower != "false":
            mssqlconfhelper.printValidationErrorMessage(self, _("{0} value must be '{1}' or '{2}'").format(self.name, self.true_expected_value, self.false_expected_value))
            return False
        return True

class IntegerSetting(MssqlRegistrySetting):
    """Generic integer setting
    """

    def __init__(self, name, environment_variable, setting_value_type, description, section, restart_required):
        super(IntegerSetting, self).__init__(name, environment_variable, setting_value_type, description, section, restart_required)

    def validateSetting(self, setting_value, validate_verb_used=False):
        return True

class TcpPortSetting(MssqlRegistrySetting):
    """TCP port setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        try:
            integers = [int(i) for i in setting_value.split(',')]
            for number in integers:
                if number < 1 or number > 65536:
                    mssqlconfhelper.printValidationErrorMessage(self, _("Port numbers must be between 1 and 65535"))
                    return False
                if (validate_verb_used == False):
                    with contextlib.closing(socket.socket(socket.AF_INET, socket.SOCK_STREAM)) as sock:
                        result = sock.connect_ex(('127.0.0.1', number))
                        if(result == 0):
                            mssqlconfhelper.printValidationErrorMessage(self, _("Port '%s' is already in use. Please use another port") % number)
                            return False
            return True
        except ValueError:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid integer") % setting_value)
            return False
        return True

class MemoryLimitSetting(MssqlRegistrySetting):
    """Memory limit setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        try:
            memorySetting = int(setting_value)
        except ValueError:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid integer") % setting_value)
            return False

        if (memorySetting < 1):
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is too small") % setting_value)
            return False

        return True

class IpAddressSetting(MssqlRegistrySetting):
    """IP address setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        return True

class DirectorySetting(MssqlRegistrySetting):
    """Backup directory setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        return validateDirectoryExists(self, setting_value)

class DirectorySettingList(MssqlRegistrySetting):
    """Colon separated directory paths setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        paths = setting_value.split(':')
        for path in paths:
            if path != '' and not validateDirectoryExists(self, path):
                return False
        return True

class TraceFlagSetting(MssqlRegistrySetting):
    """Trace flag setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        try:
            traceflagNumber = int(setting_value)
            if((1 <= traceflagNumber <= 65535) == False):
                mssqlconfhelper.printValidationErrorMessage(self, _("Traceflag '%s' should be between 1 and 65535") % traceflagNumber)
                return False
            return True
        except ValueError:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not an integer") % setting_value)
            return False

class CoreDumpTypeSetting(MssqlRegistrySetting):
    """Core dump type setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        dump_type = setting_value
        if(dump_type == "mini" or dump_type == "miniplus" or dump_type == "filtered" or dump_type == "full"):
            return True
        mssqlconfhelper.printValidationErrorMessage(self, _("coredumptype value must be mini, miniplus, filtered or full"))
        return False

class AadCertificateSetting(MssqlRegistrySetting):
    """AAD certificate setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        if not setting_value or not setting_value.lower().endswith(".pfx"):
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid .pfx file") % setting_value)
            return False
        
        return validateFileExists(self, setting_value)

class GuidSetting(MssqlRegistrySetting):
    """GUID setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        try:
            uuid_obj = UUID(setting_value, version=4)
        except ValueError:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid UUID") % setting_value)
            return False
        return str(uuid_obj) == setting_value

class AADAdminNameSetting(MssqlRegistrySetting):
    """AAD admin name setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        return len(setting_value) > 0

class AADAdminTypeSetting(MssqlRegistrySetting):
    """AAD admin type setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        if (len(setting_value) == 1):
            if ("0" == setting_value or "1" == setting_value):
                return True

        mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is neither 0 nor 1") % setting_value)
        return False

class AadEndpointSetting(MssqlRegistrySetting):
    """AAD certificate setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        return True

class TlsCertificateSetting(MssqlRegistrySetting):
    """TLS certificate setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        return validateFileExists(self, setting_value)

class TlsKeySetting(MssqlRegistrySetting):
    """TLS key setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        return validateFileExists(self, setting_value)

class TlsProtocolsSetting(MssqlRegistrySetting):
    """TLS protocols setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        protocols = re.findall(r'[\d\.]+',setting_value)
        validProtocols = [
            "1.0",
            "1.1",
            "1.2"
        ]

        if len(protocols) == 0:
            mssqlconfhelper.printValidationErrorMessage(self, _('tlsprotocols must have at least one protocol version specified'))

        for protocol in protocols:
            if protocol not in validProtocols:
                mssqlconfhelper.printValidationErrorMessage(self, _("'{0}' is not a supported TLS protocol version. Valid values are {1}").format(protocol, str(validProtocols)))
                return False

        return True

class TlsCiphersSetting(MssqlRegistrySetting):
    """TLS ciphers setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        # cipher list format here: http://openssl.org/docs/man1.0.1/apps/ciphers.html
        # try to validate using the 'openssl ciphers' command
        try:
            if subprocess.call(['openssl', 'ciphers', setting_value], stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE) != 0:
                mssqlconfhelper.printValidationErrorMessage(self, _("tlsciphers value is invalid. Refer to docs.microsoft.com for cipher string usage and examples."))
                return False
            else:
                return True
        except OSError:
            # we couldn't run the command
            mssqlconfhelper.printValidationErrorMessage(self, _("Unable to validate tlsciphers. Verify that OpenSSL is installed properly."))
            return False

class LcidSetting(MssqlRegistrySetting):
    """LCID setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        if mssqlconfhelper.isValidLcid(setting_value):
            return True

        mssqlconfhelper.printValidationErrorMessage(self, _("LCID %s is not supported.") % setting_value)
        return False

class AgentErrorLoggingLevelSetting(MssqlRegistrySetting):
    """SQL Agent logging level bitmask - 1=Errors, 2=Warnings, 4=Info
    """

    def validateSetting(self, setting_value, validate_verb_used=False):

        try:
            mask = int(setting_value)
        except ValueError:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid integer") % setting_value)
            return False

        if (mask < 1 or mask > 7):
            mssqlconfhelper.printValidationErrorMessage(self, _("Logging Level mask must be between 1 and 7"))
            return False

        return True

class AgentErrorLogFileSetting(MssqlRegistrySetting):
    """SQL Agent error log file path - may or may not already exist. Directory must exist
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        (directory, file_name) = os.path.split(setting_value)
        if not file_name :
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid file") % setting_value)
            return False
        return validateDirectoryExists(self, directory)

class KerberosKeytabFileSetting(MssqlRegistrySetting):
    """Kerberos keytab file path
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        return validateFileExists(self, setting_value)

class RBinPathFileSetting(MssqlRegistrySetting):
    """R Executable Home
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        return validateFileExists(self, setting_value) and self.validateRevoScaleRInstalled(setting_value)
    
    def validateRevoScaleRInstalled(self, r_bin_path):
        rruntime = extensibilityruntime.RRuntime(r_bin_path)
        if not rruntime.get_revoscale_path():
            mssqlconfhelper.printRevoScaleRNotInstalledMessage()
        
        return True

class PythonBinPathFileSetting(MssqlRegistrySetting):
    """Python Executable Home
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        return validateFileExists(self, setting_value) and self.validateRevoScalePyInstalled(setting_value)

    def validateRevoScalePyInstalled(self, python_bin_path):
        pythonruntime = extensibilityruntime.PythonRuntime(python_bin_path)
        if not pythonruntime.get_revoscale_path():
            mssqlconfhelper.printRevoScalePyNotInstalledMessage()
        
        return True

class ADUserSetting(MssqlRegistrySetting):
    """AD user
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        if setting_value.find('@') == -1 and setting_value.find('/') == -1:
            return True
        else:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid privileged AD account. Only input the username, not the fully qualified name.") % setting_value)
            return False

class UpdateFrequencySetting(MssqlRegistrySetting):
    """Time in seconds between credential updates
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        res = True

        try:
            l = int(setting_value)

            if l > 0 and l < 2**32:
                res = True
            else:
                res = False


        except ValueError:
            res = False

        if not res:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid time in seconds between 1 and 4294967295") % setting_value)

        return res

class DomainListSetting(MssqlRegistrySetting):
    """Semicolon-separated list of domains
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        if re.search("^[a-zA-Z0-9\-\.;]+$", setting_value):
            return True
        else:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid list of domains. Provide only the domain names separated by semicolons") % setting_value)
            return False

class AgentDatabaseMailProfileSetting(MssqlRegistrySetting):
    """SQL Agent Database Mail profile name
    """

    # The name of the profile can be any string, no validation
    def validateSetting(self, setting_value, validate_verb_used=False):
        return True

class AcceptEulaSetting(MssqlRegistrySetting):
    """accepteula setting. This is set during setup by mssql-conf and is hidden from the user
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        # Since this setting is hidden from the user and is set only by mssql-conf
        # this value must be 'Y'
        if setting_value == "Y":
            return True

        return False

class AcceptEulaMlSetting(MssqlRegistrySetting):
    """accepteulaml setting. This is set during setup by mssql-conf and is hidden from the user
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        # Since this setting is hidden from the user and is set only by mssql-conf
        # this value must be 'Y'
        if setting_value == "Y":
            return True

        return False

class MasterDataFileSetting(MssqlRegistrySetting):
    """masterdatafile setting. This is set during setup by mssql-conf. Can be set later too but needs restart
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        (directory, file_name) = os.path.split(setting_value)
        if not file_name or not file_name.lower().endswith(".mdf"):
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid .mdf file") % setting_value)
            return False

        return validateDirectoryExists(self, directory)

class MasterLogFileSetting(MssqlRegistrySetting):
    """masterlogfile setting. This is set during setup by mssql-conf. Can be set later too but needs restart
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        (directory, file_name) = os.path.split(setting_value)
        if not file_name or not file_name.lower().endswith(".ldf"):
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid .ldf file") % setting_value)
            return False

        return validateDirectoryExists(self, directory)

class ErrorLogFileSetting(MssqlRegistrySetting):
    """errorlogfile setting. This is set during setup by mssql-conf. Can be set later too but needs restart
    """
    def validateSetting(self, setting_value, validate_verb_used=False):
        (directory, file_name) = os.path.split(setting_value)
        if not file_name :
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid file") % setting_value)
            return False

        return validateDirectoryExists(self, directory)

class MachineKeyFileSetting(MssqlRegistrySetting):
    """machinekeyfile setting. This is set during setup by mssql-conf. If already set, key should be moved
    """
    def validateSetting(self, setting_value, validate_verb_used=False):
        (directory, file_name) = os.path.split(setting_value)
        if not file_name:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid file") % setting_value)
            return False

        return validateDirectoryExists(self, directory)

class HeStackSizeSetting(MssqlRegistrySetting):
    """HeStack Size setting in KB
    """

    def validateSetting(self, setting_value, validate_verb_used=False):

        try:
            kb = int(setting_value)
        except ValueError:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid integer") % setting_value)
            return False

        if (kb < 128):
            mssqlconfhelper.printValidationErrorMessage(self, _("Stack size must be greater than or equal to 128KB"))
            return False

        return True

class Ipv6DnsRecordsLimitSetting(MssqlRegistrySetting):
    """Ipv6DnsRecordsLimit setting
    """

    def validateSetting(self, setting_value, validate_verb_used=False):

        try:
            ipv6DnsRecordsLimit = int(setting_value)
        except ValueError:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid integer") % setting_value)
            return False

        if (ipv6DnsRecordsLimit < 0):
            mssqlconfhelper.printValidationErrorMessage(self, _("Invalid IPv6 DNS records limit : '%d'. Valid limits must be between 0 and 5 inclusively.") % ipv6DnsRecordsLimit)
            return False

        if (ipv6DnsRecordsLimit > 5):
            mssqlconfhelper.printValidationErrorMessage(self, _("Invalid IPv6 DNS records limit : '%d'. Valid limits must be between 0 and 5 inclusively.") % ipv6DnsRecordsLimit)
            return False

        return True

class NumErrorLogsSetting(MssqlRegistrySetting):
    """
    Number of error log maintained before cycling the log.
    """

    def validateSetting(self, setting_value, validate_verb_used=False):

        try:
            num_error_logs = int(setting_value)
        except ValueError:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid integer") % setting_value)
            return False

        # Internally the engine will clamp all values to at least the minimum.
        #
        # We set an upper bound to make sure the customer doesn't shoot them selves the foot.
        # The upper bound is the same as SSMS restricts.
        #
        minimum_errorlogs = 6
        maximum_errorlogs = 99
        if (minimum_errorlogs > num_error_logs or num_error_logs > maximum_errorlogs):
            mssqlconfhelper.printValidationErrorMessage(self, _("The number of error logs must be between 6 and 99"))
            return False

        return True

class UncMapping(MssqlRegistrySetting):
    """
    Section of unc path mapping. Setting and value is provided by user.
    setting_value can only be of the format //servername/sharename or //servername/sharename/.
    """

    def validateSetting(self, setting_value, validate_verb_used=False, setting_name=""):
        val = validateDirectoryExists(self, setting_value)
        if not val:
            mssqlconfhelper.printValidationErrorMessage(self, _("UNC mapping needs to point to an existing directory"))
            return False
        key = setting_name.startswith("//")
        if not key:
            mssqlconfhelper.printValidationErrorMessage(self, _("UNC path must begin with //"))
            return False
        keyPath = setting_name.count("/")
        if keyPath == 3 or (keyPath == 4 and setting_name.endswith("/")):
            return True
        mssqlconfhelper.printValidationErrorMessage(self, _("Only server share can be mapped. UNC path should be of the format //servername/sharename or //servername/sharename/"))
        return False

class DistributedTransactionTraceValue(MssqlRegistrySetting):
    """distributed transaction trace value.
    """

    def validateSetting(self, setting_value, validate_verb_used=False):

        try:
            value = int(setting_value)
            if value < 0 or value > 255:
                mssqlconfhelper.printValidationErrorMessage(self, _("'%s' should be in the range 0 to 255") % setting_value)
                return False
        except ValueError:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid integer") % setting_value)
            return False
        return True

class DistributedTransactionIntegerValue(MssqlRegistrySetting):
    """distributed transaction LogSize value.
    """

    def validateSetting(self, setting_value, validate_verb_used=False):

        try:
            value = int(setting_value)
        except ValueError:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not a valid integer") % setting_value)
            return False
        return True

class ExternalPolicyBasedAuthorizationSetting(MssqlRegistrySetting):
    """External policy based authorization setting
       setting_format can only be of the form 'Azure Purview' or ''
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        if setting_value == "Azure Purview":
            return True
        if setting_value == "":
            return True
        return False

class PurviewPdsEndpointSetting(MssqlRegistrySetting):
    """Purview Pds endpoint setting
       setting_format can only be of the form https://<value>
    """

    def validateSetting(self, setting_value, validate_verb_used=False, setting_name=""):
        if len(setting_name) > 2048:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' exceeds maximum length of 2048 characters") % setting_value)
            return False

        result = urllib.parse.urlparse(setting_value)
        scheme = result.scheme
        domain = result.netloc

        if not scheme:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' No URL scheme specified") % setting_value)
            return False

        if not re.fullmatch(SCHEME_FORMAT, scheme):
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' URL scheme must be http(s)") % setting_value)
            return False

        if not domain:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' does not have a domain") % setting_value)
            return False

        if not re.fullmatch(DOMAIN_FORMAT, domain):
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' does not have a valid domain") % setting_value)
            return False
 
        return True

class PurviewPolicyApiVersionSetting(MssqlRegistrySetting):
    """Purview policy api version setting
	   setting_value can only be 2021-01-01-preview
    """

    def validateSetting(self, setting_value, validate_verb_used=False, setting_name=""):
        if setting_value == "2021-01-01-preview":
            return True
        if setting_value == "":
            return True

        return False

class PurviewResourceEndpointSetting(MssqlRegistrySetting):
    """Purview resource endpoint setting
       setting_format can only be of the form https://purview.azure.net
    """

    def validateSetting(self, setting_value, validate_verb_used=False, setting_name=""):
        if len(setting_name) > 2048:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' exceeds maximum length of 2048 characters") % setting_value)
            return False

        result = urllib.parse.urlparse(setting_value)
        scheme = result.scheme
        domain = result.netloc

        if not scheme:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' does not have a domain") % setting_value)
            return False

        if not re.fullmatch(SCHEME_FORMAT, scheme):
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' URL scheme must be http(s)") % setting_value)
            return False

        if not domain:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' does not have a domain") % setting_value)
            return False

        if not re.fullmatch(DOMAIN_FORMAT, domain):
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' does not have a valid domain") % setting_value)
            return False
 
        return True

class PurviewConfigSettingsSetting(MssqlRegistrySetting):
    """Configuration settings for PolicyPullFrequency and PolicyExpiryPeriod
    """

    def validateSetting(self, setting_value, validate_verb_used=False):
        if setting_value == "":
            return True

        try:
            jsonvalue = json.loads(setting_value)
        except ValueError:
            mssqlconfhelper.printValidationErrorMessage(self, _("'%s' is not valid json") % setting_value)
            return False

        return True
        
