import mssqlconfhelper
import os
import contextlib
import logging
import subprocess
import typing

logger = logging.getLogger("pyadutil")
_ = mssqlconfhelper._

# adutil is not currently being bundled with SQL until adutil is Executive Order compliant.
#
# adutilLocation = os.path.abspath(os.path.join(os.path.dirname(__file__), "../../bin/adutil"))
adutilLocation = "adutil"

def printError(text):
    """printError

    Args:
        text(str): Text to print
    """

    mssqlconfhelper._printTextInColor(text, mssqlconfhelper.bcolors.RED)
    logger.error(text)

@contextlib.contextmanager
def withAdutilLocation(newAdutilLocation):
    """ A context manager which temporarily forces callAdutil (via findAdutil) to
        return adutilLocation without inspecting the host platform.
    """
    global adutilLocation
    oldVal = adutilLocation

    try:
        adutilLocation = newAdutilLocation
        yield adutilLocation
    finally:
        adutilLocation = oldVal

def callAdutil(*args) -> typing.Optional[str]:
    """ Invokes adutil with the specified arguments. Returns stdout from command.
    """
    stdout = subprocess.PIPE
    fullArgs = [adutilLocation, "--accept-eula"] + list(args)
    argsForLogging = ', '.join("'%s'" % arg for arg in fullArgs)
    logger.info("Executing adutil: [%s].", argsForLogging)

    try:
        process = subprocess.Popen(fullArgs, stdin=subprocess.PIPE, stdout=stdout)
    except OSError:
        printError(_("adutil is not installed for the current user."))
        exit(mssqlconfhelper.errorExitCode)
    poutput, perror = process.communicate()

    if poutput is not None:
        poutput = poutput.decode("utf-8")

    if perror is not None:
        perror = perror.decode("utf-8").strip()
        if perror != "":
            logger.error("adutil printed to stderr: [%s]", perror)

    if process.returncode != mssqlconfhelper.successExitCode:
        logger.error("adutil command failed with exit code: %s and command output [%s].", process.returncode, poutput)
        raise subprocess.CalledProcessError(process.returncode, cmd=fullArgs, output=poutput)

    logger.info("adutil returned successfully with output: [%s]", poutput)
    return poutput
