""" Provides utilities for setting up active directory on Linux for MSSQL.
"""
from configparser import ConfigParser
from contextlib import contextmanager
import socket
import subprocess
import os
import re
import getpass
import pwd
import grp
import mssqlconfhelper
import ctypes
import ctypes.util
import pyadutil
import socket
import typing

# Special constants for specifying KVNO values.
#
USE_CURRENT_KVNO = "USE_CURRENT_KVNO"
USE_NEXT_KVNO = "USE_NEXT_KVNO"

# Setup common shortcuts from mssqlconfhelper
#
logger = mssqlconfhelper.logger
printError = mssqlconfhelper.printError
printException = mssqlconfhelper.printException
successExitCode = mssqlconfhelper.successExitCode
errorExitCode = mssqlconfhelper.errorExitCode
configurationFilePath = mssqlconfhelper.configurationFilePath
_ = mssqlconfhelper._

#
# Static configuration values
#
spnPrefix = "MSSQLSvc"
spnDefaultPort = "1433"
kvnoCommandFormat = "kvno %s"
keytabEncryptionTypes = ["aes256-cts-hmac-sha1-96"]
listKeytabContentsCommandFormat = "klist -kte %s"
copyKeytabEntriesPrefixCommandFormat = "echo \"rkt %s\n"
copyKeytabEntriesSuffixCommandFormat = """wkt %s
quit
\" | ktutil"""
rootKeytab = "/etc/krb5.keytab"
rootKrb5Config = "/etc/krb5.conf"
mssqlConfPassword = "MSSQL_CONF_PASSWORD"
tempCredentialCache = "MEMORY:credential-cache"
keytabSettingName = "kerberoskeytabfile"
accountSettingName = "privilegedadaccount"

class adconfig():
    """ This class is used to validate the AD setup.
    """
    def __init__(self, keytab, realm):
        """ Initializes the ad config given a keytab filename and a realm (most of the time this is the default realm).
        """
        if checkLoggedIn() != successExitCode:
            print(_("Error: Root must be logged in as an AD user to run this command. Please run 'sudo kinit <user>"))
            exit(errorExitCode)

        # Check if mssql.conf has an entry for the kerberos keytab
        #
        config = ConfigParser()
        mssqlconfhelper.readConfigFromFile(config, configurationFilePath)

        keytabSetting = mssqlconfhelper.getSettings(configurationFilePath,
                                    "network",
                                    "kerberoskeytabfile")

        if len(keytabSetting) == 0:
            print(_("Warning: No keytab specified in mssql.conf. Please run 'mssql-conf set network.kerberoskeytabfile <path>'. /etc/krb5.keytab will be used by default"))

        if keytab == "":
            if keytabSetting != {}:
                keytab = keytabSetting["kerberoskeytabfile"]
            else:
                keytab = rootKeytab

        # Check if SQL Server is running on a non-default port
        #
        portSetting = mssqlconfhelper.getSettings(configurationFilePath,
                                  "network",
                                  "tcpport")

        if len(portSetting) > 0:
            self.port = portSetting["tcpport"]
        else:
            self.port = "1433"

        # Initialize all member variables
        #
        self.defaultRealm = getDefaultRealm()
        self.keytab = keytab
        self.realm = realm

        self.rdns = ""
        self.spnDictionary = {}
        self.privilegedAccount = ""
        self.privilegedKvno = ""

    def validate(self):
        """ This function checks if the system is configured for AD
            authentication. If so, it returns successExitCode.
        """
        # Check a default realm is set
        #
        if self.defaultRealm == "":
            print(_("Error: No default realm found in /etc/krb5.conf"))
            exit(errorExitCode)

        # Check we can contact the realm
        #
        shortDefaultRealm = self.defaultRealm.split('.')[0]

        exitCode = successExitCode
        if not self.isConnectedToLdapHost(shortDefaultRealm):
            printError(_("Error: Cannot contact default realm '%s'") % (shortDefaultRealm))
            exitCode = errorExitCode

        if not self.isConnectedToLdapHost(self.defaultRealm):
            printError(_("Error: Cannot contact default realm '%s'") % (self.defaultRealm))
            exitCode = errorExitCode

        if self.realm != "":
            shortRealm = self.realm.split('.')[0]

            if not self.isConnectedToLdapHost(shortRealm):
                printError(_("Error: Cannot contact realm '%s'") % (shortRealm))
                exitCode = errorExitCode

            if not self.isConnectedToLdapHost(self.realm):
                printError(_("Error: Cannot contact realm '%s'") % (self.realm))
                exitCode = errorExitCode

        if self.realm != "":
            rdnsRealm = self.realm
        else:
            rdnsRealm = self.defaultRealm

        # Check RDNS is setup correctly
        #
        rdns, checkRDNSErr = self.checkRDNS(rdnsRealm)
        self.rdns = rdns
        if checkRDNSErr != successExitCode:
            print(_("Warning: RDNS could not resolve this host. This is not a fatal error, but should be fixed by updating your RDNS records"))

        # Check keytab is readable
        #
        if not self.checkKeytabReadable():
            printError(_("Error: Keytab %s is not readable by the mssql group or does not exist") % (self.keytab))
            exitCode = errorExitCode

        # Check the SPN entries are present in the keytab
        #
        if self.checkSPNs() != successExitCode:
            printError(_("Error: SPN entries are not present in %s or have the wrong KVNO") % (self.keytab))
            exitCode = errorExitCode

        # Check the privileged account (e.g. UPN) is present in the keytab
        #
        if self.checkPrivilegedEntry() != successExitCode:
            printError(_("Error: %s entries are not present in %s or have the wrong KVNO") % (self.privilegedAccount, self.keytab))
            exitCode = errorExitCode

        # Print configuration so user can confirm the values are what they
        # expected
        #
        self.toString()

        if exitCode != successExitCode:
            printError(_("One or more errors were detected, see console output for more details"))

        exit(exitCode)

    @classmethod
    def isConnectedToLdapHost(cls, host):
        """ Check if a host can be contacted by attempting to connect to one of the LDAP(S) ports.
            Returns True if a connection can be made and False if not.
        """

        # At least one of the following ports should be open since SQL occaisionally needs to use LDAP.
        #
        globalCatalogPorts = [3268, 3269]
        logger.info("Checking to see if this host can connect to [%s] using an AD global catalog port [%s].", host, globalCatalogPorts)

        def handleOSError(port):
            logger.exception("Could not connect to %s:%s. This may be ok as long as at least one port works with this host.", host, port)

        def handleOtherError(port):
            logger.exception("Unexpected exception occurred while trying to connect to %s:%s.", host, port)

        def isAddrInfoConnected(addrinfo, port):
            """ Checks that a particular set of addrinfo can
                have a connection opened to it. Port is passed in explicitly
                for logging.
            """
            try:
                ip_family, _, ip_proto, _, sockaddr = addrinfo
                sock = socket.socket(family=ip_family, type=socket.SOCK_STREAM, proto=ip_proto)
                sock.connect(sockaddr)
                return True
            except OSError:
                handleOSError(port)
                return False
            except Exception:
                handleOtherError(port)
                return False

        def isPortConnected(port):
            """ This helper function returns True if the input port can establish a TCP connection and
                False otherwise.
            """
            try:
                # Do DNS resolution explicitly in case they are using IPv6 for
                # connecting to the DC node. However, mandate the port since that is
                # not configurable and mandate TCP since we need to see if we can establish
                # a connection for this validation to work.
                #
                logger.debug("Looking up address info for host at port %s.", port)
                addrinfoList = socket.getaddrinfo(host, port, type=socket.SOCK_STREAM)
                logger.debug("Found IP connection info via getaddrinfo: [%s].", addrinfoList)

                return any(isAddrInfoConnected(addrinfo, port) for addrinfo in addrinfoList)
            except OSError:
                handleOSError(port)
                return False
            except Exception:
                handleOtherError(port)
                return False

        return any(isPortConnected(port) for port in globalCatalogPorts)
    
    @classmethod
    def checkRDNS(cls, realm: str) -> typing.Tuple[str, int]:
        """ Checks if the reverse DNS entries for this machine are configured. Returns the hostname
            obtained by doing RDNS as well as the exit code (hostname, exitcode).
        """
        logger.info("Checking rDNS records for SQL Server")

        # Get our IP address
        #
        fqdn = socket.getfqdn()
        ip = socket.gethostbyname(fqdn)

        logger.info("Determined FQDN to be [%(fqdn)s] and IP to be [%(ip)s] for this host", {
            "fqdn": fqdn,
            "ip": ip
        })

        # Perform reverse DNS lookup on IP
        #
        try:
            hostname = socket.gethostbyaddr(ip)
        except socket.herror:
            logger.exception("Could not perform rDNS for IP [%s]", ip)
            return fqdn, errorExitCode
        
        # Check the rDNS result is in the domain we think we are part of
        #
        rdns = hostname[0]
        logger.info("Found hostname to be [%s] using rDNS", rdns)

        matchObject = re.match(".*%s" % (realm), rdns, re.IGNORECASE)

        if matchObject != None and matchObject.group() == rdns:
            logger.info("Verified that hostname ends with realm name")
            return rdns, successExitCode
        else:
            logger.error("Hostname [%(hostname)s] does not contain realm name [%(realm)s]", {
                "hostname": rdns,
                "realm": realm
            })
            return rdns, errorExitCode

    def checkKeytabReadable(self):
        """ Check if the keytab is readable by the mssql user
        """

        logger.info("Checking if keytab [%s] is readable by the mssql group.", self.keytab)
        try:
            groupId = os.stat(self.keytab).st_gid
            logger.debug("Determined gid to be %s.", groupId)
            mssqlGroupId = grp.getgrnam("mssql").gr_gid
            logger.debug("Determined gid for mssql to be %s", mssqlGroupId)
            if groupId == mssqlGroupId:
                logger.info("Keytab has group mssql and should thus be readable by SQL.")
                return True
            else:
                logger.error("Keytab is not owned by the mssql group and thus may not be readable by SQL.")
                return False
        except OSError:
            printError(_("OS error encountered when checking if %s is readable") % (self.keytab))
            return False

    @classmethod
    def checkKeytabEntry(cls, keytab: str, entryName: str, isPrivilegedAccount: bool = False) -> typing.Tuple[int, int]:
        """ Check if the specified service / account is in the keytab and has the latest KVNO. Returns (kvno, errorCode).
        """
        logger.info("Checking that SPN/account [%(entryName)s] exists in keytab [%(keytab)s] and is valid", {
            "entryName": entryName,
            "keytab": keytab
        })

        # Valid KVNOs should be positive.
        #
        kvno = 0

        # Get the keytab entries
        #
        try:
            keytabEntries = parseKeytab(keytab)
        except KeytabParseException:
            printException(_("Could not parse keytab %(keytab)s") % {"keytab": keytab})
            return kvno, errorExitCode

        # Get the kvno for the requested entry. There is different logic for SPNs vs users because adutil does not yet
        # support looking up the kvno for an SPN.
        #
        try:
            if isPrivilegedAccount:
                kvno = getUserKvno(entryName)
            else:
                kvno = getSPNKvno(entryName)
        except:
            printException(_("Error encountered while looking up KVNO for %s") % (entryName))
            return kvno, errorExitCode
        
        for keytabEntry in keytabEntries:
            entryKvno = int(keytabEntry.kvno)
            kvnoMatches = entryKvno == kvno or entryKvno == 255
            if kvnoMatches and keytabEntry.principal.casefold() == entryName.casefold():
                logger.info("Found matching entry")
                return entryKvno, successExitCode
        
        logger.info("Could not find matching entry in keytab")
        return kvno, errorExitCode

    def checkSPNs(self):
        """ Check if the required SPNs are in the keytab
        """

        hostname = self.rdns.split('.')[0]
        realm = self.defaultRealm

        if self.realm != "":
            realm = self.realm

        # Generate all possible SPNs
        #
        validSPNs = getSPNs(hostname, realm, self.port)

        # Check if SPNs are present
        #
        for spn in validSPNs:
            kvno, ret = adconfig.checkKeytabEntry(self.keytab, spn)

            if ret == successExitCode:
                self.spnDictionary[spn] = kvno

        # Only fail if we found no valid SPNs
        #
        ret = successExitCode

        if len(self.spnDictionary) == 0:
            ret = errorExitCode

        # Print a warning if not all SPNs are present
        #
        if len(self.spnDictionary) < len(validSPNs) and len(self.spnDictionary) > 0:
            print(_("Warning: Some but not all SPNs are present"))

        return ret

    def checkPrivilegedEntry(self):
        """ Check if the privileged entry (i.e. UPN) is present in the keytab
        """

        # Get the privileged account from mssql.conf
        #
        config = ConfigParser()
        mssqlconfhelper.readConfigFromFile(config, configurationFilePath)

        privilegedAccountSetting = mssqlconfhelper.getSettings(configurationFilePath,
                                               "network",
                                               "privilegedadaccount")

        # If no privileged account specified, default to the UPN
        #
        if privilegedAccountSetting != {}:
            privilegedAccount = privilegedAccountSetting["privilegedadaccount"]
        else:
            privilegedAccount = getUPN(self.defaultRealm)

        # Check the privileged account is in the keytab
        #
        kvno, ret = self.checkKeytabEntry(self.keytab, privilegedAccount, True)

        self.privilegedKvno = kvno
        self.privilegedAccount = privilegedAccount

        return ret

    def toString(self):
        """ Print stored configuration attributes
        """

        print(_("\nDetected Configuration:"))
        print(_("Default Realm: %s") % (self.defaultRealm))

        if self.realm != "":
            print(_("Realm: %s") % (self.realm))

        print(_("Keytab: %s") % (self.keytab))
        print(_("Reverse DNS Result: %s") % (self.rdns))
        print(_("SQL Server Port: %s") % (self.port))
        print(_("Detected SPNs (SPN, KVNO):"))

        for spn in self.spnDictionary:
            print("\t(%s, %s)" % (spn, self.spnDictionary[spn]))

        print(_("Privileged Account (Name, KVNO): \n\t(%s, %s)") % (self.privilegedAccount, self.privilegedKvno))


def getUPN(domain):
    """ Generate the SPN from the hostname and check against the root keytab with the input domain.
    """

    domain = domain.lower()
    logger.info("Getting UPN from hostname of this computer for domain [%s].", domain)

    fqdn = socket.getfqdn()
    upn = fqdn.split('.')[0]
    upn = upn.strip()

    if len(upn) > 15:
        message = _("Warning: UPN of machine %s (taken from the FQDN) is greater than 15 characters.") % upn
        logger.warning(message)
        print(message)
    upn += "$"

    logger.info("Verifying that the UPN [%s] exists in the root keytab [%s].", upn, rootKeytab)
    parsedKeytab = []

    try:
        parsedKeytab = parseKeytab(rootKeytab)
    except KeytabParseException:
        message = _("Error parsing keytab %s to validate UPN %s.") % (rootKeytab, upn)
        logger.exception(message)
        logger.warning("Assuming that UPN is correct since keytab could not be parsed.")
        return upn

    def findMatches(testUpn):
        return [entry for entry in parsedKeytab if entry.principal.lower() == testUpn and entry.domain.lower() == domain]

    if len(findMatches(upn.lower())) == 0:
        # First remove $ then get last 15 characters.
        #
        shortUpn = upn[:-1][:15] + "$"
        logger.warning("No matches found for upn [%s] in root keytab. Trying 15 character upn [%s].", upn, shortUpn)

        if len(findMatches(shortUpn.lower())) == 0:
            # Do not exit early here since it possible that we have just incorrectly parsed the keytab.
            #
            printError(_("Could not determine UPN since machine hostname is not present in root keytab %s.") % rootKeytab)
        else:
            logger.info("Using shortened UPN since it is present in to the root keytab.")
            upn = shortUpn

    logger.info("Determined UPN to be [%s] from fqdn [%s]", upn, fqdn)
    return upn

def checkLoggedIn():
    """ Checked if the user has kinit'ed
    """

    logger.info("Checking to see if user has run kinit.")
    ret = errorExitCode

    try:
        klistCommand = "klist -s"
        process = subprocess.Popen(klistCommand.split(), stderr=subprocess.PIPE)
        out, err = process.communicate()
        ret = process.returncode
        logger.info("klist command returned code: %s.", ret)
        logger.info("klist command had stderr [%s].", err)
    except OSError:
        printError(_("OS error encountered when checking if root is logged into active directory"))

    return ret

class Krb5Str(ctypes.c_char_p):
    """ This class is used so that we can free any strings properly. If using ctypes.c_char_p directly as
        a return type, ctypes will only return the python string and not the C string.
    """
    pass

class Krb5Exception(Exception):
    """ Exception which is thrown due to an unrecoverable errror with the krb5 library.
    """
    def __init__(self, *args, **vargs):
        """ Pass any arguments to the super class (Exception).
        """
        super(Krb5Exception, self).__init__(*args, **vargs)

def discoverKrb5Lib():
    """ Attempts to discover a shared library for krb5
    """
    logger.info("Attempting to dynamically discover shared library for krb5")

    try:
        libName = ctypes.util.find_library("krb5")
        if libName is None:
            logger.info("No shared library for krb5")
        else:
            logger.info("Determined krb5 shared library to be [%s]", libName)
    except:
        logger.exception("Error trying to discover shared library dynamically for krb5")
        libName = None

    return libName

def loadKrb5Lib():
    """ Loads a krb5 shared library (defaulting to libkrb5.so). Returns (library, libname)
    """
    defaultLibName ="libkrb5.so" 
    makeKrb5LoadingException = lambda: Krb5Exception("Could not load a krb5 shared library")

    try:
        result = ctypes.CDLL(defaultLibName)
        logger.info("Successfully loaded %s", defaultLibName)
        libName = defaultLibName
    except OSError as e1:
        logger.exception("Could not load the default krb5 library %s, attempting to discover another one dynamically", defaultLibName)

        try:
            dynLibName = discoverKrb5Lib()

            if dynLibName is None:
                logger.error("Could not find a krb5 library")
                raise makeKrb5LoadingException() from e1

            result = ctypes.CDLL(dynLibName)
            logger.warning("Successfully loaded non-default krb5 library: [%s]", dynLibName)
            libName = dynLibName
        except OSError as e2:
            raise makeKrb5LoadingException() from e2
    
    return result, libName

class CKrb5:
    """ A python wrapper around krb5 C functions. It should be called within a context manager else the caller
        of the constructor should ensure that close is called. While the functions are generally stateless in python,
        this class may not be threadsafe since it shares a single krb5_context_t.
        This library should only throw Krb5Exception, any other exceptions thrown would be a bug with this class.
    """
    def __init__(self):
        """ Loads libkrb5.so and creates a krb5_context_t for use with other methods of this class.
            A Krb5Exception will be thrown if libkrb5.so does not exist, if context initializing fails,
            or in case of an unexpected runtime exception.
        """
        self.krb5Context = None
        krb5, krb5LibName = loadKrb5Lib()
        self.krb5 = krb5
        self.krb5LibName = krb5LibName

        try:
            logger.debug("Initializing krb5 context.")
            krb5Context = ctypes.c_void_p()
            func = self.krb5.krb5_init_context
            func.restype = ctypes.c_int32
            initRetVal = func(ctypes.pointer(krb5Context))
            self.krb5Context = krb5Context

            if initRetVal != successExitCode:
                # Can't get error message since that requires a context.
                #
                message = "Initializing krb5 context returned non-zero exit code: %s." % initRetVal
                logger.error(message)
                raise Krb5Exception(message)
        except Exception as e:
            # Ensure any exception thrown is a Krb5Exception.
            #
            if isinstance(e, Krb5Exception):
                raise
            else:
                raise Krb5Exception("Error while initializing krb5 context.") from e
        logger.debug("Initialized krb5 context.")

    def __enter__(self):
        """ On entrance of a context block. Returns self.
        """
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        """ Called on exit of context block. Calls self.close().
        """
        self.close()

    def _callCVoidFunc(self, func, *args):
        """ Calls the input dynamic C function with the input args and sets the return type to void.
        """
        func.restype = None
        func(*args)

    def close(self):
        """ Cleans up this object including freeing the C krb5 context. This function
            is idempotent since it checks whether the krb5 context has already been freed.
        """

        if self.krb5Context is None:
            return
        try:
            logger.debug("Cleaning up krb5 context.")
            self._callCVoidFunc(self.krb5.krb5_free_context, self.krb5Context)
            self.krb5Context = None
            logger.debug("Cleaned up krb5 context.")
        except Exception:
            logger.exception("Could not cleanup kerberos context.")

    def getKrb5ErrorMessage(self, retval):
        """ Gets the error message for a particular return code from the krb5 library.
            This function implicitly uses the krb5 context and should NOT be called before that
            context is initialized or after that context is freed.
        """
        logger.info("Getting krb5 error message for return value %s.", retval)
        defaultMessage = _("Unknown error")
        result = defaultMessage

        try:
            # krb5 guarantees that this must be a valid null terminated string.
            #
            func = self.krb5.krb5_get_error_message
            func.restype = Krb5Str
            cErrStr = func(self.krb5Context, ctypes.c_int32(retval))
        except Exception:
            logger.exception("Could not get error message for return value %s.", retval)
            return result

        try:
            result = cErrStr.value.decode("utf-8")
            logger.info("Retrieved error string [%s] for return value %s.", result, retval)
            logger.debug("Freeing error message C string.")
            self._callCVoidFunc(self.krb5.krb5_free_error_message, self.krb5Context, cErrStr)
            logger.debug("Freed error message C string.")
        except Exception:
            logger.exception("Could not free error message string.")
        return result

    def getDefaultRealm(self):
        """ Gets the default realm using the krb5 library.
        """
        logger.debug("Calling krb5_get_default_realm.")
        cDefaultRealm = ctypes.c_char_p()
        func = self.krb5.krb5_get_default_realm
        func.restype = ctypes.c_int32
        getDefaultRealmRetVal = func(self.krb5Context, ctypes.pointer(cDefaultRealm))

        try:
            if getDefaultRealmRetVal != successExitCode:
                libError = self.getKrb5ErrorMessage(getDefaultRealmRetVal)
                errorMessage = "Could not get default realm. The library error message was [%s] and the error code was %s." % (libError, getDefaultRealmRetVal)
                raise Krb5Exception(errorMessage)
            defaultRealm = cDefaultRealm.value
            logger.info("Found default realm: [%s].", defaultRealm)

            if defaultRealm is None:
                raise Krb5Exception("Default realm was null")

            return defaultRealm.decode("utf-8")
        except Exception as e:
            if isinstance(e, Krb5Exception):
                raise
            else:
                raise Krb5Exception("Unexpected error (likely python bug) while getting default realm.") from e
        finally:
            logger.debug("Cleaning up C string for default realm.")

            if cDefaultRealm.value is None:
                # This usually happens if getting the default realm failed for some reason.
                #
                logger.debug("Ignoring cleanup of null pointer value.")
            else:
                try:
                    self._callCVoidFunc(self.krb5.krb5_free_default_realm, self.krb5Context, cDefaultRealm)
                    logger.debug("Freed default realm string.")
                except Exception:
                    logger.exception("Could not free default realm string.")

def getDefaultRealm():
    """ Find the default realm using the C krb5 library.
    """
    # Log here about getting default realm so that viewers of logs know why libkrb.so is being used before CKrb5.getDefaultRealm is called.
    #
    logger.info("Getting default realm.")

    try:
        with CKrb5() as krb5:
            return krb5.getDefaultRealm()
    except Krb5Exception:
        logger.exception("Could not get default realm using krb5.")
        printError(_("Could not get default realm."))
        return ""

def validateInputKvno(inputKvno: typing.Union[int, str]):
    """ Validates inputKvno is USE_CURRENT_KVNO, USE_NEXT_KVNO, or a positive integer.
    """
    if not (isinstance(inputKvno, int) or isinstance(inputKvno, str)):
        # Python doesn't enforce type checking and we want to be really certain that this parameter is
        # correctly typed.
        #
        printError(_("KVNO [%s] is neither an integer nor a string") % inputKvno)
        exit(errorExitCode)
    elif isinstance(inputKvno, int) and inputKvno <= 0:
        # Assume that this case can be reached
        #
        printError(_("If KVNO is given explicitly, it must be positive yet it was actually %d") % inputKvno)
        exit(errorExitCode)
    elif isinstance(inputKvno, str) and inputKvno not in [USE_CURRENT_KVNO, USE_NEXT_KVNO]:
        # This is also unlikely to ever be reached unless mssql-conf has a bug.
        #
        printError(_("Unexpected special internal constant for KVNO '%s'. This is likely a bug with mssql-conf itself") % inputKvno)
        exit(errorExitCode)

def setupADKeytab(keytab: str, inputUser: str, *, interactively: bool = True, inputKvno: typing.Union[int, str] = USE_CURRENT_KVNO):
    """ Entry point for auto-generating the AD keytab for SQL Server. The interactively parameter indicates
        whether password can be queried for interactively. If kvno is integer, it should be positive and be the
        KVNO value to be used in the keytab. If kvno is a string, it should either be USE_CURRENT_KVNO or
        USE_NEXT_KVNO and in both cases the current KVNO will be looked up from AD and either that value or
        that value plus one will be used for the keytab.
    """

    logger.info("Setting up keytab for mssql for user [%s] at path [%s] and input KNVO: [%s]", inputUser, keytab, inputKvno)
    validateInputKvno(inputKvno)

    originalSettings = None
    try:
        originalSettings = getADSettings()
        logger.info("Read old AD settings: [%s]", originalSettings)
    except:
        logger.exception("Could not read old settings for AD")

    # Login is required otherwise keytabs will not contain correct KVNO.
    #
    if checkLoggedIn() != successExitCode:
        printError(_("Error: Root must be logged in as an AD user to run this command. Please run 'sudo kinit <user>"))
        exit(errorExitCode)
    logger.info("User has already run 'sudo kinit'")

    # Get the hostname
    #
    hostname = socket.gethostname().strip()
    logger.info("Determined full hostname of this machine to be [%s]", hostname)
    hostnameSplit = hostname.find('.')

    if hostnameSplit != -1:
        hostname = hostname[:hostnameSplit]
    logger.debug("Determined rightmost component of hostname to be [%s].", hostname)

    # Get the domain
    #
    domain = getDefaultRealm()

    if domain == "":
        printError(_("Error: Default realm must be set in /etc/krb5.conf"))
        exit(errorExitCode)

    domain = domain.lower()
    logger.info("Using AD domain [%s] for setup.", domain)

    try:
        usingLatestKvno = inputKvno == USE_CURRENT_KVNO
        userInfo = setupADUser(inputUser, domain, interactively=interactively, usingLatestKvno=usingLatestKvno)
    except Exception:
        printException(_("Error while validating or setting up user in AD."))
        exit(errorExitCode)
    
    # Determine value for KVNO
    #
    if isinstance(inputKvno, int):
        kvno = inputKvno
        logger.info("Using user specified KVNO %d", kvno)
    else:
        try:
            userWithDomain = "%s@%s" % (userInfo.user, userInfo.domain)
            latestKvno = getUserKvno(userWithDomain)
        except Exception:
            printException(_("Error looking up current KVNO"))
            exit(errorExitCode)
        
        logger.info("Determined current KVNO to be %d", latestKvno)
        
        if inputKvno == USE_NEXT_KVNO:
            kvno = latestKvno + 1
        else:
            # inputKvno == USE_CURRENT_KVNO
            #
            kvno = latestKvno
        
        logger.info("Using %d for KVNO", kvno)

    # Get all potential SPNs
    #
    spnList = getSPNs(hostname, domain)

    userSpns = [spn for spn in spnList if addSpnToUser(userInfo.user, spn)]

    if len(userSpns) == 0:
        printError(_("Error: No registered SPNs found. Please register SPNs then re-run this script"))
        exit(errorExitCode)
    else:
        logger.info("Using %s spns out of %s.", len(userSpns), len(spnList))

    with withPasswordForAdutil(userInfo.password):
        logger.info("Adding user to keytab.")

        if not createKeytab(keytab, userInfo.user, kvno):
            printError(_("Could not add base user to keytab."))
            exit(errorExitCode)

        if not usingLatestKvno:
            logger.info("Skipping validation of user credentials since KVNO %d may not be current", kvno)
        elif not validADCredentials(userInfo.user, keytab=keytab):
            # This really only possible if we were unable to validate that the user exists early on since we do not validate the password in that case.
            # Otherwise, getting here likely indicates a bug in mssql-conf.
            #
            printError(_("Unable to login as user %s@%s using keytab %s.") % (userInfo.user, keytab))
            exit(errorExitCode)

        logger.info("Added user to keytab.")

        # Add SPNs to keytab
        #
        spnsInKeytab = 0

        for spn in userSpns:
            if createKeytab(keytab, spn, kvno):
                # Can't just use validADCredentials since you can't kinit directly as an SPN.
                #
                logger.info("Added SPN %s to the keytab.", spn)
                spnsInKeytab += 1
            else:
                printError(_("Failed to add SPN %s to the keytab %s.") % (spn, keytab))

        if spnsInKeytab == len(userSpns):
            logger.info("Added all SPNs to the keytab.")
        elif spnsInKeytab > 0:
            logger.warning("Only added %s SPNs to the keytab.", spnsInKeytab)
        else:
            printError(_("Could not add any SPNs to the keytab."))
            exit(errorExitCode)

    # Set owner of keytab
    #
    try:
        logger.info("Setting owner of keytab file to mssql.")
        mssqlUID = pwd.getpwnam("mssql")
        os.chown(keytab, mssqlUID.pw_uid, mssqlUID.pw_gid)
        logger.info("Set owner of keytab file to mssql.")
    except KeyError:
        logger.warning("Cannot set owner of keytab file to mssql (mssql user probably does not exist).")
        print(_("Warning: User 'mssql' does not exist. Auto-generated keytab "
                "still owned by root. Please change ownership to the user SQL "
                "Server runs as (chown mssql:mssql %s") % (keytab))

    # Set keytab path in mssql.conf
    #
    absKeytab = os.path.abspath(keytab)
    logger.info("Setting keytab file in mssql settings to the keytab's absolute file path [%s].", absKeytab)
    mssqlconfhelper.setSettingByName("network", keytabSettingName, absKeytab)

    # Set privileged user in mssql.conf (and clear it if no privileged user)
    #
    mssqlconfhelper.setSettingByName("network", accountSettingName, userInfo.user)

    needRestart = True
    try:
        newSettings = getADSettings()
        needRestart = originalSettings is not None and newSettings == originalSettings
    except:
        logger.exception("Could not read new AD settings so must assume restart is required")

    if needRestart:
        msg1 = _("SQL Server needs to be restarted in order to adopt the new AD configuration, please run")
        msg2 = _("'systemctl restart mssql-server.service'.")
        logger.info(msg1)
        logger.info(msg2)
        print(msg1)
        print(msg2)
    else:
        logger.info("Since no AD settings in mssql.conf changed during this operation, restart of SQL is NOT required.")

def getADSettings(configurationFilePath: typing.Optional[str] = None) -> typing.FrozenSet[typing.Tuple[str, typing.Optional[str]]]:
    """ Returns a frozenset of tuples for the settings relevant for AD (i.e. priveledged user and keytab location).
        This is suitable for detecting if changes have occurred which require a SQL restart. If no file path is specified,
        settings are read from mssqlconfhelper.configurationFilePath.
    """
    if configurationFilePath is None:
        configurationFilePath = mssqlconfhelper.configurationFilePath
    settingsDictionary = mssqlconfhelper.getSettings(configurationFilePath, "network")
    makeTuple = lambda name: (name, settingsDictionary.get(name))
    return frozenset([makeTuple(keytabSettingName), makeTuple(accountSettingName)])

class ADUserInfo:
    """ This class is a data holder for various useful information about users in AD.
        usingUPN indicates that the user is a machine rather than a normal user.
    """
    def __init__(self, user, domain, password, usingUPN):
        self.user = user
        self.domain = domain
        self.password = password
        self.usingUPN = usingUPN

    def __repr__(self):
        # Password is redacted for obvious reasons
        #
        return "ADUserInfo(user=[%s], domain=[%s], usingUPN=[%s])" % (self.user, self.domain, self.usingUPN)

    def __str__(self):
        return repr(self)

    def _asTuple(self):
        return (self.user, self.domain, self.password, self.usingUPN)

    def __eq__(self, other):
        if not isinstance(other, ADUserInfo):
            return False
        return self._asTuple() == other._asTuple()

    def __hash__(self):
        return hash(self._asTuple())

def getUserKvno(user: str) -> int:
    """ Returns the latest KVNO for the input user. This is guaranteed to be positive.
    """
    logger.info("Looking up latest KVNO for [%s]", user)
    kvnoStr = pyadutil.callAdutil("account", "kvno", user)

    if kvnoStr is None:
        raise Exception("Call to adutil failed to return a string when looking up latest KVNO")
    
    try:
        kvno = int(kvnoStr)
    except Exception as e:
        errorMessage = "Could not parse KVNO '%s' as an integer" % kvnoStr
        raise Exception(errorMessage) from e
    
    if kvno <= 0:
        errorMessage = "Expected KVNO to be positive, but it was actually %s" % kvno
        raise Exception(errorMessage)
    
    return kvno

def getSPNKvno(spn: str) -> int:
    """ Gets the KVNO for an SPN. Returns (kvno, errorCode).
    """
    logger.info("Looking up KVNO for [%s]", spn)

    getKvnoCommand = "kvno %s" % (spn)
    logger.info("Using 'kvno' command to lookup current KVNO for SPN, command is [%s]", getKvnoCommand)

    try:
        kvnoProcess = subprocess.Popen(getKvnoCommand.split(), stdout=subprocess.PIPE, stderr=open(os.devnull, 'w'))
        kvnoOutput, error = kvnoProcess.communicate()
        ret = kvnoProcess.returncode
    except OSError:
        errorMessage = _("OS error encountered when finding the KVNO for %s") % (spn)
        raise Exception(errorMessage)
    
    if ret != 0:
        errorStr = "Error looking up kvno for SPN with error code %s" % (ret)
        logger.error(errorStr)
        if error is not None:
            logger.error(error.decode("utf-8", "replace"))
        raise Exception(errorStr)
    
    try:
        kvnoOutputStr = kvnoOutput.decode("utf-8")
    except UnicodeDecodeError as e:
        raise Exception("Could not decode 'kvno' command output [%s]" % (kvnoOutput)) from e
    
    logger.info("'kvno' command returned output [%s]", kvnoOutputStr)
    kvnoParts = kvnoOutputStr.split("=")

    if len(kvnoParts) > 1:
        # Use regex to search for the account in the keytab. The
        # kvno can be either the current kvno or 255
        #
        return int(kvnoParts[1].strip())
    else:
        raise Exception("Could not parse 'kvno' command output")

def setupADUser(inputUser, domain, *, interactively: bool = True, usingLatestKvno: bool = True) -> ADUserInfo:
    """ Returns an ADUserInfo object with all fields set or else throws an exception. If this function returns, the
        user is generally guaranteed to exist and the password is valid. The only exception is when the user's existence
        cannot be determined one way or the other in which case this function assumes that the user already exists and
        that the password is valid.
    """
    logger.info("Setting up inputUser [%s] on domain [%s] with interactively? [%s]", inputUser, domain, interactively)

    # If user is not specified, use the UPN instead
    #
    if inputUser is None or inputUser == "":
        user = getUPN(domain)
        logger.info("No user was specified, using upn [%s] for setting up AD keytab.", user)
        usingUPN = True
    else:
        user = inputUser
        logger.info("Using a non-UPN user.")
        usingUPN = False

    # We should be able to do this check regardless if it is a machine or a regular user.
    # This variable is True/False/None.
    #
    userAlreadyExists = checkIfADUserExists(user)

    if usingUPN and userAlreadyExists == False:
        # Should never get here, probably a bug with this script if we do reach here.
        #
        message = _("If using machine account, machine account should already exist in AD.")
        printError(message)
        raise Exception(message)

    # Get the password to use for inserting keytab entries. Need != since userAlreadyExists can be None
    #
    password = getPasswordForUser(user, domain, interactively=interactively, doubleCheck=(not usingLatestKvno))

    if password is None:
        message = _("Password must be specified through environment variable %s or interactively.") % mssqlConfPassword
        printError(message)
        raise Exception(message)

    if userAlreadyExists is None:
        logger.warning("Assuming that user already exists since it could not be determined using adutil.")
    elif usingLatestKvno and userAlreadyExists:
        logger.info("Since user already exists and we are using the latest KVNO, verify that password is correct.")
        userIsValid = validADCredentials(user, password=password)

        if userIsValid:
            logger.info("User already exists and password is correct.")
        else:
            message = _("Password is not correct.")
            printError(message)

            if usingUPN:
                print(_("If password is not known for machine in AD, you will need to change machine password to a known value."))
            raise Exception(message)
    elif userAlreadyExists:
        logger.info("User exists, but skipping password validation since not using the latest KVNO value")
    else:
        # User does not exist
        #
        logger.info("User does not yet exist.")
        userDoesNotExistMessage = _("User %s@%s does not exist and cannot be created in non-interactive mode.") % (user, domain)

        if not usingLatestKvno:
            invalidKvnoToCreateUserMessage = _("User %s does not yet exist and KVNO (key version) must be the latest version (i.e. without --kvno or --use-next-kvno) to create user") % user
            printError(invalidKvnoToCreateUserMessage)
            raise Exception(invalidKvnoToCreateUserMessage)

        if not interactively:
            printError(userDoesNotExistMessage)
            raise Exception(userDoesNotExistMessage)
        yesOrNoStr = input(_("User %s does not exist, would you like to create it %s? (yes/no):") % (user, user)).strip().lower()
        yesValue = _("yes")

        if yesOrNoStr == yesValue:
            logger.info("Script runner indicated that they wish to create user [%s] by entering [%s].", user, yesValue)

            if not createUserInAD(user, domain, password):
                message = _("Could not create user %s@%s.") % (user, domain)
                printError(message)
                raise Exception(message)
        else:
            logger.info("Script runner entered [%s] which is not '%s' so do NOT create user.", yesOrNoStr, yesValue)
            creationOptOutMessage = _("User %s@%s does not exist and must be created before AD can be setup for mssql.") % (user, domain)
            printError(creationOptOutMessage)
            raise Exception(creationOptOutMessage)

    logger.info("User now exists and should have the correct password unless user existence could not be determined or KVNO is not current.")
    return ADUserInfo(user, domain, password, usingUPN)

def validADCredentials(principal, password=None, keytab=None):
    """ Validates either password or keytab for the input principal depending on whether a password or keytab is specified.
        Exactly one of the password or keytab arguments should be given or else an exception will be thrown.
        Returns True if the credentials are valid else returns False.
    """
    # The following should never happen during normal execution of mssql-conf.
    #
    if password is None and keytab is None:
        raise ValueError("Either password or keytab must be supplied.")
    elif password is not None and keytab is not None:
        raise ValueError("Exactly one of password or keytab must be not None.")

    if password is not None:
        logger.info("Validating password for principal [%s].", principal)
    else:
        logger.info("Validating keytab [%s] for principal [%s].", keytab, principal)

    # Validate credentials by using a temporarty credential cache to kinit and setting a low expiry time of 1 minute.
    #
    baseArgs = ["kinit", "-c", tempCredentialCache, "-l 1m", principal]

    if keytab is not None:
        args = baseArgs + ["-kt", keytab]
    else:
        args = baseArgs
    logger.info("Validating credentials by running running subprocess command [%s].", " ".join(args))
    process = subprocess.Popen(args, stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE)

    # Even if password is None, that is the same as sendning nothing through the pipe which is what we want for the keytab case.
    #
    if password is not None:
        # As of python3, communicate requires a bytes object rather than a string.
        #
        password = password.encode("utf-8")
    process.communicate(password)

    if process.returncode == errorExitCode:
        printError(_("Incorrect credentials for AD user: %s.") % principal)
        return False
    elif process.returncode != successExitCode:
        logger.error("kinit returned non-zero exit code %s.", process.returncode)
        printError(_("Could not validate credentials for AD user: %s, ensure kinit is installed correctly.") % principal)
        return False
    else:
        logger.info("kinit ran successfully, credentials are correct.")
        return True

def getSPNs(hostname, domain, port=None):
    """ Generate all possible SPNs
    """

    spnList = []

    # Get the port if not present.
    #
    if port is None:
        logger.info("Finding default mssql port since no port is specified.")
        port = spnDefaultPort
        portSetting = mssqlconfhelper.getSettings(configurationFilePath,
                                "network",
                                "tcpport")

        if len(portSetting) > 0:
            port = portSetting["tcpport"]
            logger.info("Using port from mssql settings %s.", port)
        else:
            logger.info("Using default port %s since none is specified in settings", spnDefaultPort)
    logger.info("Determining SPNs for hostname [%s] on domain [%s] with port %s.", hostname, domain, port)

    # Generate all potential SPNs
    #
    spnList.append("%s/%s.%s:%s" % (spnPrefix, hostname, domain, port))
    spnList.append("%s/%s.%s" % (spnPrefix, hostname, domain))
    spnList.append("%s/%s:%s" % (spnPrefix, hostname, port))
    spnList.append("%s/%s" % (spnPrefix, hostname))
    spnStringList = ", ".join("'%s'" % spn for spn in spnList)
    logger.info("Determined SPNs for mssql to be [%s].", spnStringList)

    return spnList

@contextmanager
def withPasswordForAdutil(password):
    """ This function sets the ADUTIL_ACCOUNT_PWD environment variable which is used by adutil in place of setting it interactively.
        It is expected to be invoked using the 'with' statement. After leaving the context of the 'with' block, the ADUTIL_ACCOUNT_PWD
        environment variable will be set to its previous value.
    """
    logger.info("Temporarily setting ADUTIL_ACCOUNT_PWD as an environment variable.")
    passwordVarName = "ADUTIL_ACCOUNT_PWD"

    if passwordVarName in os.environ:
        oldPassword = os.environ[passwordVarName]
    else:
        oldPassword = None
    try:
        os.environ[passwordVarName] = password
        logger.info("Set ADUTIL_ACCOUNT_PWD environment variable to input value.")
        yield
    finally:
        logger.info("Cleaning up ADUTIL_ACCOUNT_PWD environment variable.")

        if oldPassword is not None:
            logger.info("There was a previous value for ADUTIL_ACCOUNT_PWD, setting to original value.")
            os.environ[passwordVarName] = oldPassword
            logger.info("Set ADUTIL_ACCOUNT_PWD to original value.")
        else:
            # First set to empty string then delete in case os.unsetenv is not supported for platform.
            #
            logger.info("No previous value for ADUTIL_ACCOUNT_PWD so deleting ADUTIL_ACCOUNT_PWD from environment variables.")
            os.environ[passwordVarName] = ""
            del os.environ[passwordVarName]
            logger.info("Deleted ADUTIL_ACCOUNT_PWD from environment variables.")

def createUserInAD(user, domain, password):
    """ Creates the specified user with the specified password in AD. Returns True if user creation succeeds and False otherwise.
    """
    logger.info("Creating user %s@%s.", user, domain)

    with withPasswordForAdutil(password):
        domainComponents = ",".join("DC=" + dpart.upper() for dpart in domain.split("."))
        distname = "CN=%s,CN=users,%s" % (user, domainComponents)

        try:
            pyadutil.callAdutil("user", "create", "--name", user, "--distname", distname)
            logger.info("Successfully created user.")
            return True
        except subprocess.CalledProcessError:
            printError(_("Could not create user %s") % user)
            return False


def getPasswordForUser(user, domain, interactively=True, doubleCheck=False):
    """ Returns the password for the user on the domain. Returns None if no password is indicated by the environment variable MSSQL_CONF_PASSWORD and
        in non-interactive mode. Will only query for password if MSSQL_CONF_PASSWORD is blank or not present. Trailing whitespace will NOT be trimmed on this variable.
    """
    logger.info("Getting password for %s@%s.", user, domain)

    if mssqlConfPassword in os.environ:
        logger.info("Password has been specified using %s.", mssqlConfPassword)
        return os.environ[mssqlConfPassword]
    if not interactively:
        logger.error("Cannot get password since not in interactive mode.")
        return None
    password = getpass.getpass(_("%s@%s's password: ") % (user, domain))

    if doubleCheck:
        password2 = getpass.getpass(_("Confirm %s@%s's password: ") % (user, domain))
        if password != password2:
            printError("Passwords do not match.")
            return None

    logger.info("Retrieved password interactively.")
    return password

def checkIfADUserExists(user):
    """ Checks to see the input user already exists in AD. Returns True if user exists, False if user can be determined to not exist, and None
        if an error occurs while determining if the user exists.
    """
    logger.info("Checking if user [%s] exists in AD.", user)

    try:
        maybeUserExistsString = pyadutil.callAdutil("account", "exists", user)

        if maybeUserExistsString is None:
            logger.error("'adutil account exists' returned no output")
            return None

        userExistsString = maybeUserExistsString.strip().lower()
        result = userExistsString == "true"

        if result:
            logger.info("User already exists.")
        else:
            logger.info("User does not yet exist.")
        return result
    except subprocess.CalledProcessError:
        printException(_("Could not determine if user %s already exists") % user)
        return None

def addSpnToUser(user, spn):
    """ Registers specified spn to input user
    """
    logger.info("Adding spn [%s] to user [%s].", spn, user)

    try:
        pyadutil.callAdutil("spn", "add", "-n", user, "-s", spn)
        logger.info("Successfully added spn to user.")
        return True
    except subprocess.CalledProcessError:
        printError(_("Error ocurred while adding spn %s to user %s") % (spn, user))
        return False

def createKeytab(keytab: str, principal: str, kvno: int) -> bool:
    """ Creates a keytab with the given principal. Returns True if
        the principal was added to the keytab else returns False.
    """

    try:
        enctypeCSV = ','.join(keytabEncryptionTypes)
        logger.info("Adding principal [%s] to keytab at [%s] with encryption types [%s].", principal, keytab, enctypeCSV)
        pyadutil.callAdutil("keytab", "create",
            "--path", keytab,
            "--principal", principal,
            "--enctype", enctypeCSV,
            "--kvno", str(kvno))
        logger.info("Added principal to keytab.")
        return True
    except subprocess.CalledProcessError:
        printException(_("Error ocurred while adding principal %s to keytab %s") % (principal, keytab))
        return False

class KeytabEntry:
    """ This is a parsed version of an entry in a krb5 keytab.
    """
    def __init__(self, kvno: str, principal: str, domain: str, encType: str, slot: int):
        self.kvno = kvno
        self.principal = principal
        self.domain = domain
        self.encType = encType
        self.slot = slot

    def __repr__(self):
        return "KeytabEntry(kvno='%s', principal='%s', domain='%s', encType='%s', slot=%s)" % (self.kvno, self.principal, self.domain, self.encType, self.slot)

class KeytabParseException(Exception):
    """ General exception thrown when parsing of a keytab fails.
    """
    def __init__(self, *args, **vargs):
        super(KeytabParseException, self).__init__(*args, **vargs)

def parseKeytab(keytab):
    """ Parses keytab into a list of KeytabEntry if successful otherwise throw KeytabParseException.
    """

    # Get all the entries in the keytab
    #
    logger.info("Parsing keytab at [%s].", keytab)
    listKeytabContentsCommand = listKeytabContentsCommandFormat % (keytab)

    try:
        logger.info("Running command to list keytab entries: [%s].", listKeytabContentsCommand)
        process = subprocess.Popen(listKeytabContentsCommand.split(), stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        logger.info("Successfully ran command to list keytab entries.")
        keytabContents, err = process.communicate()

        if err is not None:
            err = err.decode("utf-8").strip()

        if err is not None and err != "":
            logger.info("klist command printed to stderr: [%s].", err)

        if process.returncode != successExitCode:
            message = _("klist command returned non-zero exit code %s for keytab [%s].") % (process.returncode, keytab)
            printError(message)
            raise KeytabParseException(message)
    except OSError as e:
        message = "OS error while parsing keytab [%s].".format(keytab)
        raise KeytabParseException(message) from e
    except Exception as e:
        if isinstance(e, KeytabParseException):
            raise
        else:
            raise KeytabParseException("Unexpected error occurred while running klist") from e

    # Surround parsing code with general try/catch in case something goes wrong with parsing.
    #
    try:
        keytabContents = keytabContents.decode("utf-8")

        # Get each line in the keytab, except the first three (keytab name, column
        #  names and a horizontal line). We can assume keytabContents is present since exception handling rethrows.
        #
        keytabLines = keytabContents.split('\n')[3:]
        splitLines = [line.strip().split(' ') for line in keytabLines]

        def parseLine(lineContents, slot):
            # Get entry's KVNO, name, and encryption type
            #
            entryKvno = lineContents[0]
            parsedPrincipal = lineContents[3].split('@')
            entryPrincipal = parsedPrincipal[0]
            entryDomain = parsedPrincipal[1]
            entryEncType = lineContents[4].strip('()')
            result = KeytabEntry(kvno=entryKvno, principal=entryPrincipal, domain=entryDomain, encType=entryEncType, slot=slot)
            logger.debug("Found keytab entry: %s.", result)
            return result

        result = [parseLine(lineContents, slot) for slot, lineContents in enumerate(splitLines, start=1) if len(lineContents) >= 5]
        logger.info("Found [%s] non-blank entries in keytab.", len(result))
        return result
    except Exception as e:
        message = "Unexpected error occurred while parsing keytab [{0}].".format(keytab)
        raise KeytabParseException(message) from e
